//---------------------------------------------------------------------------
//   DIB Library
//         Copyright(c)1998-2000, O.Mizno     All rights reserved.
//
//     Dib.h
//       DibStream class
//---------------------------------------------------------------------------

#ifndef _DIB_H
#define _DIB_H

#include <stdlib.h>


struct DibBGRA
{
	union {
		struct {
			BYTE b,g,r,a;
		};
		BYTE col[4];
	};

	inline DibBGRA() { }
	inline DibBGRA(BYTE _b,BYTE _g,BYTE _r) { b=_b; g=_g; r=_r; a=0; }
	inline DibBGRA(BYTE _b,BYTE _g,BYTE _r,BYTE _a) { b=_b; g=_g; r=_r; a=_a; }
	inline DibBGRA(DWORD dw) {
		b = (BYTE)(dw&0xFF);
		g = (BYTE)((dw>>8)&0xFF);
		r = (BYTE)((dw>>16)&0xFF);
		a = (BYTE)((dw>>24)&0xFF);
	}
	inline DibBGRA(BYTE *pb) { b=pb[0]; g=pb[1]; r=pb[2]; a=pb[3]; }
};

struct DibBGR
{
	union {
		struct {
			BYTE b,g,r;
		};
		BYTE col[3];
	};

	inline DibBGR() { }
	inline DibBGR(BYTE _b,BYTE _g,BYTE _r) {b=_b; g=_g; r=_r; }
	inline DibBGR(DWORD dw) {
		b = (BYTE)(dw&0xFF);
		g = (BYTE)((dw>>8)&0xFF);
		r = (BYTE)((dw>>16)&0xFF);
	}
	inline DibBGR(BYTE *pb) { b=pb[0]; g=pb[1]; r=pb[2]; }
};

// DibStream::SetDib(SHMemType,SBMemType)
#define DIB_MEMORYTYPE_NONE         0
#define DIB_MEMORYTYPE_LINK         0
#define DIB_MEMORYTYPE_HEAP         1
#define DIB_MEMORYTYPE_LOCALLOCKED  2
#define DIB_MEMORYTYPE_GLOBALLOCKED 3
#define DIB_MEMORYTYPE_LOCALFIXED   4
#define DIB_MEMORYTYPE_GLOBALFIXED  5

// DibStream::LoadFromFile(LoadBit)
#define DIB_LOAD_1   0x01
#define DIB_LOAD_4   0x02
#define DIB_LOAD_8   0x04
#define DIB_LOAD_16  0x08
#define DIB_LOAD_24  0x10
#define DIB_LOAD_32  0x20
#define DIB_LOAD_ALL 0x3F

// DibStream::ChangeBpp(Flag)
#define DIB_CHANGEBPP_1_DIFFUSION  0
#define DIB_CHANGEBPP_4_MONO       0
#define DIB_CHANGEBPP_4_OPTIMIZE   2
#define DIB_CHANGEBPP_8_MONO       0
#define DIB_CHANGEBPP_8_DITHER     1
#define DIB_CHANGEBPP_8_OPTIMIZE   2
#define DIB_CHANGEBPP_16_DITHER    1

#define DIB_CHANGEBPP_MASK                0xFFFF
#define DIB_CHANGEBPP_OPTIMIZE_MASK       0x00FF0000
#define DIB_CHANGEBPP_OPTIMIZE_NUMBER(x)  ((x) << 16)


class DibStream
{
friend class DibStream;
protected:
	ULONG RefCount;
	union {
		LPVOID Header;
		LPBITMAPINFO BMPInfo;
		LPBITMAPINFOHEADER BMPHeader;
	};
	union {
		LPVOID Buffer;
		LPBYTE bBuffer;
	};
	DWORD HeaderMemoryType, BufferMemoryType;

	void HeaderFree(void);
	void BufferFree(void);

	LPVOID CreateHeader(int Width, int Height, int Bpp);
	LPVOID CreateBuffer(BITMAPINFOHEADER *BMPInfo);
	bool SetBitmapInfo(BITMAPINFOHEADER &BMPInfo, int Width, int Height, int Bpp);

	BYTE *Create8Bit(void);
	DibBGR *Create24Bit(void);
	DibBGRA *Create32Bit(void);
	bool ChangeTo1Bit(DWORD Flag);
	bool ChangeTo4Bit(DWORD Flag);
	bool ChangeTo8Bit(DWORD Flag);
	bool ChangeTo16Bit(DWORD Flag);
	bool ChangeTo24Bit(DWORD Flag);
	bool ChangeTo32Bit(DWORD Flag);
	bool MedianCut4(DWORD flag);
	bool MedianCut8(DWORD flag);

public:
	DibStream();
	virtual ~DibStream();
	ULONG AddRef(void);
	ULONG Release(void);
	inline ULONG GetRefCount(void){return RefCount;}

	// inline functions
	inline LPBITMAPINFO GetInfo(void) { return BMPInfo; }
	inline LPBITMAPINFOHEADER GetHeader(void) { return BMPHeader; }
	inline LPVOID GetPtr(void) { return Buffer; }
	inline LPVOID GetBuffer(void) { return Buffer; }

	inline int GetWidth(void)  { return (int)BMPHeader->biWidth; }
	inline int GetHeight(void) { return (int)abs(BMPHeader->biHeight); }
	inline int GetWidthBytes(void) { return GetPitch(); }
	inline int GetBpp(void) { return (int)BMPHeader->biBitCount; }
	inline int GetBitCount(void) { return GetBpp(); }
	inline int GetCompression(void){ return (int)(BMPHeader->biCompression); }
	inline int GetNumColors(void) {
		return (BMPHeader->biClrUsed == 0 && BMPHeader->biBitCount <= 8
			? (int)(1 << (int)BMPHeader->biBitCount) : (int)BMPHeader->biClrUsed);
	}
	inline RGBQUAD *GetPalette(void)
		{ return (RGBQUAD *)((LPBYTE)Header + (int)BMPHeader->biSize); }
	inline RGBQUAD *GetColors(void)	{ return GetPalette(); }
	inline int GetImageSize(void) {
		return (BMPHeader->biSizeImage == 0
			? (GetWidthBytes() * GetHeight()) : BMPHeader->biSizeImage);
	}
	inline int GetSizeImage(void) { return GetImageSize(); }
	inline BOOL IsBottomUp(void) { return (BMPHeader->biHeight > 0); }
	inline BOOL IsTopDown(void)  { return (BMPHeader->biHeight < 0); }
	inline BYTE *ScanLine(int y) {
		int pitch = GetPitch();
		if(IsTopDown()) return &bBuffer[pitch*y];
		return &bBuffer[pitch*(GetHeight()-y-1)];
	}

	int GetPitch(void);
	BYTE *Pixel(int x, int y);
	DWORD GetPixel(int x, int y);

	void Clear(void);
	DibStream *Clone(void);
	bool SetDib(LPVOID SourceHeader, DWORD SHMemType, LPVOID SourceBuffer, DWORD SBMemType);
	bool SetBuffer(LPVOID SourceBuffer, DWORD SBMemType);
	inline DWORD GetHeaderMemoryType(void) { return HeaderMemoryType; }
	inline DWORD GetBufferMemoryType(void) { return BufferMemoryType; }
	inline void SnatchHeaderMemory(void) { HeaderMemoryType = DIB_MEMORYTYPE_LINK; }
	inline void SnatchBufferMemory(void) { BufferMemoryType = DIB_MEMORYTYPE_LINK; }
	bool Copy(LPVOID SourceHeader, LPVOID SourceBuffer);

	DibStream *CreateAlphaDib(void);
	bool SetAlphaPlane(DibStream *AlphaDib);

	bool LoadFromFile(LPCSTR FileName, DWORD LoadBit=DIB_LOAD_ALL);
	bool SaveToFile(LPCSTR FileName);

	bool Create(int Width, int Height, int Bpp);
	bool ChangeBpp(int Bpp, DWORD Flag=0);
	void SetMonoPalette(void);
	void UpDown(bool TopDown);

	HPALETTE CreatePalette(void);
	void Draw(HDC hdc, int X, int Y);
	void StretchDraw(HDC hdc, int X, int Y, int W, int H);
	void StretchDrawRect(HDC hdc,
		int DstX, int DstY, int DstW, int DstH,
		int SrcX, int SrcY, int SrcW, int SrcH);
	void DrawWithPalette(HDC hdc, int X, int Y);

	inline void Draw(HDC hdc, POINT Dst){
		Draw(hdc, Dst.x, Dst.y);
	}
	inline void DrawRect(HDC hdc, int DstX, int DstY, int SrcX, int SrcY, int W, int H){
		StretchDrawRect(hdc, DstX, DstY, W, H, SrcX, SrcY, W, H);
	}
	inline void DrawRect(HDC hdc, POINT Dst, POINT Src, int W, int H){
		DrawRect(hdc, Dst.x, Dst.y, Src.x, Src.y, W, H);
	}
	inline void StretchDrawRect(HDC hdc, POINT DstLT, POINT DstRB, POINT SrcLT, POINT SrcRB){
		StretchDrawRect(hdc,
			DstLT.x, DstLT.y, DstRB.x-DstLT.x+1, DstRB.y-DstLT.y+1,
			SrcLT.x, SrcLT.y, SrcRB.x-SrcLT.x+1, SrcRB.y-SrcLT.y+1);
	}
	inline void StretchDrawRect(HDC hdc, RECT Dst, RECT Src){
		StretchDrawRect(hdc,
			Dst.left, Dst.top, Dst.right-Dst.left, Dst.bottom-Dst.top,
			Src.left, Src.top, Src.right-Src.left, Src.bottom-Src.top);
	}
	inline void DrawWithPalette(HDC hdc, POINT Dest){
		DrawWithPalette(hdc, Dest.x, Dest.y);
	}

	bool Resize(int w, int h);
	bool ChangeSize(int w, int h, DWORD BackColor);
	bool ClipCut(int x, int y, int w, int h);
	DibStream *ClipClone(int x, int y, int w, int h);

	bool InvertTone(void);
	bool Brighten(float val);
	bool Scroll(int x, int y);

	bool ConvertBGR2RGB();
};


#endif _DIB_H

