/********************************************************************************/
/*                                                                              */
/*		Metasequoia.cpp															*/
/*                                                                              */
/********************************************************************************/

#include "stdafx.h"
#include "Centigrade.h"
#include "Strings.h"


#define COMCHECK(i,s) afs->chkcom(s,i)
#define CMPSTR(i,s) afs->chkstr(i,s)

#define LOADERMQO_SUCCEEDED       0
#define LOADERMQO_ERROR          -1
#define LOADERMQO_EOF            -2
#define LOADERMQO_HEADER_ERR     -3
#define LOADERMQO_FORMAT_ERR     -4
#define LOADERMQO_TRIAL_NOISE    -5

/********************************************************************************/
/*                                                                              */
/*		LoaderMQONX															*/
/*                                                                              */
/********************************************************************************/

class LoaderMQO
{
protected:
	CCentigrade *centigrade;
	FILE *fp;
	AsciiFileString *afs;

public:
	int material_number;

protected:
	BOOL CheckHeader(void);
	BOOL IgnoreUnknown(int string_count);
	BOOL IgnoreBinary(int string_count);
	BOOL MainLoop(void);

	BOOL LoadMaterial(void);
	BOOL LoadObject(CModel* model);
	BOOL LoadVertex(CModel* model);
	BOOL LoadFace(CModel* model);

public:
	LoaderMQO(CCentigrade *pt,FILE *handle);
	virtual ~LoaderMQO();

	int Load(void);
};

/********************************************************************************/
/*                                                                              */
/*		CCentigrade::LoadMQOFile												*/
/*                                                                              */
/********************************************************************************/

bool CCentigrade::LoadMQOFile(const char *file_name)
{
	int ret_code;
	size_t first_model = GetModelsSize();

	FILE *fp = fopen(file_name, "rb");
	if(fp == NULL) return NULL;

	LoaderMQO *loader = new LoaderMQO(this,fp);

	loader->material_number = material_number;

	ret_code = loader->Load();

	delete loader;

	fclose(fp);

	// }eÃZbg
	if ( !materials.size() )
	{
		materials.push_back(CMaterial(material_number));	
		material_number++;
	}
	SetMaterials();

	CalcNormals();

	return (ret_code == LOADERMQO_SUCCEEDED);
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::LoaderMQO													*/
/*                                                                              */
/********************************************************************************/

LoaderMQO::LoaderMQO(CCentigrade *pt,FILE *handle)
{
	centigrade = pt;

	fp = handle;

	afs = new AsciiFileString(fp);
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::~LoaderMQO													*/
/*                                                                              */
/********************************************************************************/

LoaderMQO::~LoaderMQO()
{
	delete afs;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::CheckHeader													*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::CheckHeader(void)
{
	int sc;

	//check header
	sc = afs->GetLine();
	if(!afs->chkcom("Metasequoia",2) || !afs->chkstr(1,"Document"))
	{
		throw(LOADERMQO_HEADER_ERR);
	}

	//check format
	sc = afs->GetLine();
	if(!afs->chkcom("Format",4) || !afs->chkstr(1,"Text")
	|| !afs->chkstr(2,"Ver") || afs->ToDouble(3)!=1.0)
	{
		throw(LOADERMQO_FORMAT_ERR);
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::IgnoreBinary													*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::IgnoreBinary(int string_count)
{
	long oldpos, newpos;
	long offset = afs->ToInt(string_count-2);

	oldpos = ftell(fp);
	fseek(fp, offset, SEEK_CUR);
	newpos = ftell(fp);

	if(newpos - oldpos != offset)
	{
		throw(LOADERMQO_ERROR);
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::IgnoreUnknown												*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::IgnoreUnknown(int string_count)
{
	if(string_count >= 1 && CMPSTR(string_count-1,"{"))
	{   //pass unknown group
		for(int comc=1; comc>0; )
		{
			string_count = afs->GetLine();
			if(string_count == EOF)
			{
				throw(LOADERMQO_EOF);
			}
			else if(string_count >= 1)
			{
				if(CMPSTR(string_count-1,"{"))
				{
					comc++;
				}
				else if(CMPSTR(string_count-1,"}"))
				{
					comc--;
				}
				else if( string_count>=3
				&& afs->CheckString(string_count-3,"[")
				&& afs->CheckString(string_count-1,"]") )
				{
					IgnoreBinary(string_count-2);
				}
			}
		}
	}
	else if( string_count>=3 && afs->CheckString(string_count-3,"[") && afs->CheckString(string_count-1,"]") )
	{
		IgnoreBinary(string_count-2);
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::MainLoop														*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::MainLoop(void)
{
	for(;;)
	{
		int sc = afs->GetLine();
		if(sc == EOF) break;
		if(sc == 0) continue;

		if( afs->chkcom("TrialNoise",2) )
		{
			throw(LOADERMQO_ERROR);
		}
		else if( afs->chkcom("Material",2) )
		{
			if( !afs->chkstr(sc-1,"{") )
			{
				throw(LOADERMQO_ERROR);
			}
			if( !LoadMaterial() )
			{
				throw(LOADERMQO_ERROR);
			}
		}
		else if( afs->chkcom("Object",2) )
		{
			if(!afs->chkstr(sc-1,"{"))
			{
				throw(LOADERMQO_ERROR);
			}

			CModel* model;
			int result = 0;

			// f1₷
			centigrade->ResizeModels((int)(centigrade->GetModelsSize()+1));
			model = centigrade->GetModelsBack();

			try
			{
				if(sc >= 3)
				{
					model->SetName(afs->GetString(1));
				}
				if( !LoadObject(model) )
				{
					throw(LOADERMQO_ERROR);
				}
				CModel model2 = centigrade->GetModel((int)(centigrade->GetModelsSize()-1));
				if ( !model->Freeze(&model2) )
				{
					centigrade->PopBackModel();
					continue;
				}
				// e_ʂ̎擾
				model->FacesVertexBelongsTo();
			}
			catch(int err_code)
			{
				result = err_code;
			}
			if(result)
			{
				throw(result);
			}
		}
		else
		{
			IgnoreUnknown(sc);
		}
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::Load															*/
/*                                                                              */
/********************************************************************************/

int LoaderMQO::Load(void)
{
	try
	{
		if( !CheckHeader() )
		{
			throw(LOADERMQO_ERROR);
		}
		if( !MainLoop() )
		{
			throw(LOADERMQO_ERROR);
		}
	}
	catch(int err_code)
	{
		return err_code;
	}

	return LOADERMQO_SUCCEEDED;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::LoadMaterial													*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::LoadMaterial(void)
{
	int i,scn;

	for(;;)
	{
		CMaterial mat;

		int sc = afs->GetLine();
		if(sc == EOF)
		{
			throw(LOADERMQO_EOF);
		}
		if(sc == 0) continue;

		if(COMCHECK(1,"}")) break;

		mat.SetName(afs->GetString(0));

		for(scn=1; scn<sc;)
		{
			if(CMPSTR(scn,"col") && scn+7<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+6,")")) break;
				mat.SetRGBA(afs->ToFloat(scn+2),afs->ToFloat(scn+3),
							afs->ToFloat(scn+4),afs->ToFloat(scn+5));
				scn += 7;
			}
			else if(CMPSTR(scn,"shader") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetShader(afs->ToInt(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"dif") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetDiffuse(afs->ToFloat(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"amb") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetAmbient(afs->ToFloat(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"spc") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetSpecular(afs->ToFloat(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"emi") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetEmissive(afs->ToFloat(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"power") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetPower(afs->ToFloat(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"tex") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetTextureName(afs->GetString(scn+2));
				scn += 4;
				// eNX`̃Zbg
				mat.LoadTextureGL();
			}
			else if(CMPSTR(scn,"aplane") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetAlphaName(afs->GetString(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn,"bump") && scn+4<=sc)
			{
				if(!CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")")) break;
				mat.SetBumpName(afs->GetString(scn+2));
				scn += 4;
			}
			else if(CMPSTR(scn+1,"("))
			{
				for(i=2; i+scn<sc; i++)
				{
					if(CMPSTR(scn+i,")")) break;
				}
				scn += i;
			}
			else
			{
				break;
			}
		}

		mat.SetNumber(centigrade->GetMaterialNumber());
		centigrade->AddMaterialNumber(1);

		centigrade->PushBackMaterial(mat);
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::LoadObject													*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::LoadObject(CModel* model)
{
	for(;;)
	{
		int sc = afs->GetLine();
		if(sc == EOF)
		{
			throw(LOADERMQO_EOF);
		}
		if(COMCHECK(1,"}"))
		{
			break;
		}
		else if(COMCHECK(2,"facet"))
		{
			model->SetFacet(afs->ToFloat(1));
		}
		else if(COMCHECK(2,"mirror_axis"))
		{
			model->SetMirrorAxis(afs->ToInt(1));
		}
		else if(COMCHECK(3,"vertex"))
		{
			if(!CMPSTR(sc-1,"{")) return FALSE;
			if(!LoadVertex(model)) return FALSE;
		}
		else if(COMCHECK(2,"face"))
		{
			if(!CMPSTR(sc-1,"{")) return FALSE;
			if(!LoadFace(model)) return FALSE;
		}
		else
		{
			IgnoreUnknown(sc);
		}
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::LoadVertex													*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::LoadVertex(CModel* model)
{
	for(;;)
	{
		int sc = afs->GetLine();
		if (sc == EOF)
		{
			throw(LOADERMQO_EOF);
		}
		if( COMCHECK(1,"}")) break;

		if(sc < 3) continue;

		model->PushBackVertex(CVertex(afs->ToFloat(0),afs->ToFloat(1),afs->ToFloat(2)));
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		LoaderMQO::LoadFace														*/
/*                                                                              */
/********************************************************************************/

BOOL LoaderMQO::LoadFace(CModel* model)
{
	int scn;
	int vc;

	for(;;)
	{
		CFace face;
		int sc = afs->GetLine();
		if( sc == EOF )
		{
			throw(LOADERMQO_EOF);
		}
		if( sc == 0 ) continue;

		if ( COMCHECK(1,"}") ) break;

		vc = afs->ToInt(0); // vertex count
		scn = 1;

		if ( CMPSTR(scn,"V") && scn+vc+3 <= sc )
		{
			if ( !CMPSTR(scn+1,"(") || !CMPSTR(scn+vc+2,")") ) continue;
			if ( vc >= 1 ) face.PushBackIndex(afs->ToInt(scn+2));
			if ( vc >= 2 ) face.PushBackIndex(afs->ToInt(scn+3));
			if ( vc >= 3 ) face.PushBackIndex(afs->ToInt(scn+4));
			if ( vc >= 4 ) face.PushBackIndex(afs->ToInt(scn+5));
			scn += vc+3;
		}
		else throw(LOADERMQO_ERROR);

		if ( CMPSTR(scn,"M") && scn+4 <= sc )
		{
			if ( !CMPSTR(scn+1,"(") || !CMPSTR(scn+3,")") ) continue;
			face.SetMaterialNumber(afs->ToInt(scn+2) + material_number);
			scn += 4;
		}
		else
		{
			face.SetMaterialNumber(material_number);
		}
		if ( CMPSTR(scn,"UV") && scn+vc*2+3 <= sc )
		{
			if ( !CMPSTR(scn+1,"(") || !CMPSTR(scn+vc*2+2,")") ) continue;

			if ( vc >= 1 ) { face.PushBackUV(CUV(afs->ToFloat(scn+2),afs->ToFloat(scn+3))); }
			if ( vc >= 2 ) { face.PushBackUV(CUV(afs->ToFloat(scn+4),afs->ToFloat(scn+5))); }
			if ( vc >= 3 ) { face.PushBackUV(CUV(afs->ToFloat(scn+6),afs->ToFloat(scn+7))); }
			if ( vc >= 4 ) { face.PushBackUV(CUV(afs->ToFloat(scn+8),afs->ToFloat(scn+9))); }
			scn += vc*2+3;
		}

		model->PushBackFace(face);
	}

	return TRUE;
}

/********************************************************************************/
/*                                                                              */
/*		e_ʂ̐ݒ													*/
/*                                                                              */
/********************************************************************************/

void CModel::FacesVertexBelongsTo()
{
	int f,v,size;

	size = (int)faces.size();

	for( vector<CVertex>::iterator vertex = vertices.begin(); vertex != vertices.end(); ++vertex )
	{
		vertex->ClearHaveFaces();
	}

	for ( f = 0; f < size; ++f )
	{
		CFace *face = &faces[f];
		for ( v = 0; v < face->GetIndexSize(); ++v )
		{
			vertices[face->GetIndex(v)].PushBackHaveFaces((int)f);
		}
	}
}

/********************************************************************************/
/*                                                                              */
/*		SĂ̖ʂ3̒_ɂA~[t[Y							*/
/*                                                                              */
/********************************************************************************/

bool CModel::Freeze(CModel *model)
{
	CFace face;
	int f,v;
	int f_size = (int)model->faces.size();
	int v_size = (int)model->vertices.size();

	faces.clear();

	for ( f = 0; f < f_size; ++f )
	{
		faces.push_back(model->faces[f]);
	}

	f_size = (int)faces.size();

	switch ( mirror_axis )
	{
	case 0:
		break;
	case 1:	// X
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(-pos.GetX(),pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 2:	// Y
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),-pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 3:	// XY
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(-pos.GetX(),pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		v_size = (int)vertices.size();
		f_size = (int)faces.size();
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),-pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 4:	// Z
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),pos.GetY(),-pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 5:	// XZ
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(-pos.GetX(),pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		v_size = (int)vertices.size();
		f_size = (int)faces.size();
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),pos.GetY(),-pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 6:	// YZ
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),-pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		v_size = (int)vertices.size();
		f_size = (int)faces.size();
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),pos.GetY(),-pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	case 7:	// XYZ
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(-pos.GetX(),pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		v_size = (int)vertices.size();
		f_size = (int)faces.size();
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),-pos.GetY(),pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		v_size = (int)vertices.size();
		f_size = (int)faces.size();
		for ( v = 0; v < v_size; ++v )
		{
			CVector& pos = vertices[v].GetPos();
			vertices.push_back(CVertex(pos.GetX(),pos.GetY(),-pos.GetZ()));
		}
		for ( f = 0; f < f_size; ++f )
		{
			faces.push_back(face.GetMirrorFace(faces[f],v_size));
		}
		break;
	}

	if ( !faces.size() ) return false;

	return true;
}

/********************************************************************************/
/*                                                                              */
/*		ʂ̖																*/
/*                                                                              */
/********************************************************************************/

CFace CFace::GetMirrorFace(const CFace& pt,int size)
{
	int i,s;

	material = pt.material;

	normal = pt.normal;

	s = (int)pt.index.size();
	index.clear();
	for ( i = s-1; i >= 0; --i ) index.push_back(pt.index[i]+size);

	s = (int)pt.uv.size();
	uv.clear();
	for ( i = s-1; i >= 0; --i ) uv.push_back(pt.uv[i]);

	return *this;
}
