var _lastTime = 0;
var _elapsed = 0;
var _num = 0;
var _camera = new Matrix3D();
var _fov = 45.0;
const MAX_NUM = 10000;
const uniformBufferSize = 4 * 16 * (4 + 32); // 4byte * 4x4 matrix * (4 + 32)
const OFFSET_SIZE = uniformBufferSize; //256の倍数

async function initWebGPU(canvas) {
  if (!navigator.gpu) {
    throw Error('WebGPU not supported.');
  }
  const adapter = await navigator.gpu.requestAdapter();
  if (!adapter) {
    throw Error('Could not request WebGPU adapter.');
  }
  _device = await adapter.requestDevice();
  _canvas = document.getElementById(canvas);
  _context = _canvas.getContext('webgpu');
  _presentationFormat = navigator.gpu.getPreferredCanvasFormat();
  _context.configure({
    device: _device,
    format: _presentationFormat,
    alphaMode: 'premultiplied',
  });
  // create a render pipeline
  _pipeline = setPipeline(vertWGSL,fragWGSL);
  _pipelineTexture = setPipeline(vertWGSL,fragTextureWGSL);
  _pipelineBone = setPipeline(vertBoneWGSL,fragWGSL);
  _pipelineBoneTexture = setPipeline(vertBoneWGSL,fragTextureWGSL);
  _uniformBuffer = _device.createBuffer({
    size: uniformBufferSize + OFFSET_SIZE * MAX_NUM,
    usage: GPUBufferUsage.UNIFORM | GPUBufferUsage.COPY_DST,
  });
  window.addEventListener("resize", resize, false);
	resize();
  init();
  requestAnimationFrame(everyFrame.bind(everyFrame));
}

function setPipeline(vertexWGSL,fragmentWGSL) {
  const pipeline = _device.createRenderPipeline({
  layout: 'auto',
  vertex: {
    module: _device.createShaderModule({
      code: vertexWGSL,
    }),
    entryPoint: 'main',
    buffers: [{
     // 頂点バッファの属性を指定します。
      attributes: [
        {// position
          shaderLocation: 0, // @location(0) in vertex shader
          offset: 0,
          format: 'float32x4',
        },{// normal
          shaderLocation: 1,
          offset: 4*4,
          format: 'float32x3',
        },{// color
          shaderLocation: 2,
          offset: 4*(4+3),
          format: 'float32x4',
        },{// specular
          shaderLocation: 3,
          offset: 4*(4+3+4),
          format: 'float32',
        },{// uv
          shaderLocation: 4,
          offset: 4*(4+3+4+1),
          format: 'float32x2',
        },{// bone
          shaderLocation: 5,
          offset: 4*(4+3+4+1+2),
          format: 'float32',
        },
      ],
      arrayStride: 4*(4+3+4+1+2+1),
      stepMode: "vertex",
    },],
  },
  fragment: {
    module: _device.createShaderModule({
      code: fragmentWGSL,
    }),
    entryPoint: 'main',
    targets: [// 0
      { // @location(0) in fragment shader
        format: _presentationFormat,
      },
    ],
  },
  primitive: {
    topology: 'triangle-list',
  },
  depthStencil: {
    depthWriteEnabled: true,
    depthCompare: 'less',
    format: 'depth24plus',
  },
});
  return pipeline;
}

function everyFrame() {
  frameTime();
  _commandEncoder = _device.createCommandEncoder();
  const textureView = _context.getCurrentTexture().createView();
  _renderPassDescriptor = {
    colorAttachments: [
      {
        view: textureView,
        clearValue: { r: 0.0, g: 0.0, b: 0.0, a: 0.0 },
        loadOp: 'clear',
        storeOp: 'store',
      },
    ],
    depthStencilAttachment: {
      view: _depthTexture.createView(),
      depthClearValue: 1.0,
      depthLoadOp: 'clear',
      depthStoreOp: 'store',
    },
  };
  const passEncoder = _commandEncoder.beginRenderPass(_renderPassDescriptor);
  draw(passEncoder);
  passEncoder.end();
  _device.queue.submit([_commandEncoder.finish()]);
  requestAnimationFrame(everyFrame.bind(everyFrame));
}

function frameTime() {
	const timeNow = new Date().getTime();
	if (_lastTime != 0) {
		_elapsed = (timeNow - _lastTime) / 1000.0;
	}
  if ( _elapsed > 1/2 ) _elapsed = 1/2; 
	_lastTime = timeNow;
}

function resize() {
//  _canvas.width = window.innerWidth;
//  _canvas.height = window.innerHeight;
  _depthTexture = _device.createTexture({
    size: [_canvas.width, _canvas.height],
    format: 'depth24plus',
    usage: GPUTextureUsage.RENDER_ATTACHMENT,
  });
}

function degree2radian(degree) {
	return degree * Math.PI / 180;
}

function radian2degree(radian) {
	return radian * 180 / Math.PI;
}
//キャンバスにおけるマウスの位置を取得
function getPos(event) {
  const clientRect = _canvas.getBoundingClientRect();
  if ( 'ontouchend' in document ) {
    const touch = event.changedTouches.item(0);
    const x = touch.clientX - clientRect.left;
    const y = touch.clientY - clientRect.top;
    return new Vector2D(x,y);
  } else {
    const x = event.clientX - clientRect.left;
    const y = event.clientY - clientRect.top;
    return new Vector2D(x,y);
  }
}

async function loadTexture(texture) {
  const response = await fetch(texture);
  const imageBitmap = await createImageBitmap(await response.blob());

  let imageTexture = _device.createTexture({
    size: [imageBitmap.width, imageBitmap.height, 1],
    format: 'rgba8unorm',
    usage:
      GPUTextureUsage.TEXTURE_BINDING |
      GPUTextureUsage.COPY_DST |
      GPUTextureUsage.RENDER_ATTACHMENT,
  });
  _device.queue.copyExternalImageToTexture(
    { source: imageBitmap },
    { texture: imageTexture },
    [imageBitmap.width, imageBitmap.height]
  );
  return imageTexture;
}

