const vertWGSL = `
struct Uniforms {
  projectionMatrix : mat4x4f,
  viewMatrix : mat4x4f,
  worldMatrix : mat4x4f,
  normalMatrix : mat4x4f,
}
@group(0) @binding(0) var<uniform> uniforms : Uniforms;

struct VertexOutput {
  @builtin(position) position : vec4f,
  @location(0) normal : vec3f,
  @location(1) color : vec4f,
  @location(2) specular : f32,
  @location(3) uv : vec2f
}

@vertex
fn main(
  @location(0) position: vec4f,
  @location(1) normal: vec3f,
  @location(2) color: vec4f,
  @location(3) specular: f32,
  @location(4) uv : vec2f,
  @location(5) bone : f32
) -> VertexOutput {
  var output : VertexOutput;
  output.position = uniforms.projectionMatrix *
    uniforms.viewMatrix * uniforms.worldMatrix * position;
  output.normal = (uniforms.normalMatrix * vec4f(normal,1)).xyz;
  output.color = color;
  output.specular = specular;
  output.uv = uv;
  return output;
}
`;
const vertBoneWGSL = `
struct Uniforms {
  projectionMatrix : mat4x4f,
  viewMatrix : mat4x4f,
  worldMatrix : mat4x4f,
  normalMatrix : mat4x4f,
  boneMatrix : array<mat4x4f, 24>,
}
@group(0) @binding(0) var<uniform> uniforms : Uniforms;

struct VertexOutput {
  @builtin(position) position : vec4f,
  @location(0) normal : vec3f,
  @location(1) color : vec4f,
  @location(2) specular : f32,
  @location(3) uv : vec2f
}

@vertex
fn main(
  @location(0) position: vec4f,
  @location(1) normal: vec3f,
  @location(2) color: vec4f,
  @location(3) specular: f32,
  @location(4) uv : vec2f,
  @location(5) bone : f32
) -> VertexOutput {
  var output : VertexOutput;
  var matrix = uniforms.boneMatrix[u32(bone)];
  output.position = uniforms.projectionMatrix *
    uniforms.viewMatrix * uniforms.worldMatrix * matrix * position;
  var n = uniforms.normalMatrix;
  var n3x3 = mat3x3f(n[0].xyz,n[1].xyz,n[2].xyz);
  var m3x3 = mat3x3f(matrix[0].xyz,matrix[1].xyz,matrix[2].xyz);
  output.normal = n3x3 * m3x3 * normal;
  output.color = color;
  output.specular = specular;
  output.uv = uv;
  return output;
}
`;
const fragWGSL = `
struct VertexOutput {
  @builtin(position) position : vec4f,
  @location(0) normal : vec3f,
  @location(1) color : vec4f,
  @location(2) specular : f32,
  @location(3) uv : vec2f
}
@fragment
fn main(fragData: VertexOutput) -> @location(0) vec4f {
  var lightDirection = normalize(vec3f(-1.0,-2.0,-4.0));
  var normal = normalize(fragData.normal);
  var diffuseLightWeighting = (1+dot(normal,lightDirection))/2;
  var reflect = normalize(2.0*diffuseLightWeighting*normal-lightDirection);
  var spc = pow(clamp(dot(reflect,lightDirection),0.0,1.0),5.0);
  var specular = vec4(fragData.specular,fragData.specular,fragData.specular,1);
  var color = fragData.color*clamp(diffuseLightWeighting,0.0,1.0)+spc*specular;
  return vec4f(color.rgb,1.0);
}
`;
const fragTextureWGSL = `
@group(0) @binding(1) var mySampler : sampler;
@group(0) @binding(2) var myTexture : texture_2d<f32>;

struct VertexOutput {
  @builtin(position) position : vec4f,
  @location(0) normal : vec3f,
  @location(1) color : vec4f,
  @location(2) specular : f32,
  @location(3) uv : vec2f
}
@fragment
fn main(fragData: VertexOutput) -> @location(0) vec4f {
  var lightDirection = normalize(vec3f(-1.0,-2.0,-4.0));
  var normal = normalize(fragData.normal);
  var diffuseLightWeighting = (1+dot(normal,lightDirection))/2;
  var texColor = textureSample(myTexture, mySampler, fragData.uv);
  var emi = fragData.color.r;
  var color = texColor*clamp(emi+diffuseLightWeighting,0.0,1.0);
  return vec4f(color.rgb,1.0);
}
`;
